
// =========================================
// 1. 全局常量与状态管理 
// =========================================

// UI组件注册表
const uiList = [];

// 事件注册表 
const UI_EVENTS = {};

// CSS类名映射常量 
const CSS_CLASSES = {
  primaryBtn: 'primary-btn',
  textInput: 'text-input',
  customRadio: 'custom-radio',
  customCheckbox: 'custom-checkbox',
  inputLabel: 'input-label',
  // 下拉框相关
  dropdownContainer: 'dropdown-container',
  dropdownSelected: 'dropdown-selected',
  dropdownOptions: 'dropdown-options',
  dropdownArrow: 'dropdown-arrow',
  arrowUp: 'arrow-up',
  optionSelected: 'option-selected',
  dropdownActive: 'dropdown-active',
  transferItem: 'transfer-item',
  // 列表相关
  simpleList: 'simple-list-container',
  listSelected: 'list-selected'
};

/**
 * 注册组件到全局列表
 * @param {string} id 组件ID
 * @param {string} type 组件类型
 */
function joinList(id, type) {
  uiList.push({ id: id, type: type });
}

/**
 * 辅助函数：判断数组是否相等
 */
const isArrEqual = (arr1, arr2) => {
  if (!Array.isArray(arr1) || !Array.isArray(arr2)) return false;
  return arr1.length === arr2.length && arr1.every((ele) => arr2.includes(ele));
};

/**
 * 辅助函数：转义JSON特殊字符
 */
function escapeControlCharacters(jsonString) {
  const escapeMap = {
    '\b': '\\b', '\t': '\\t', '\n': '\\n', '\f': '\\f', '\r': '\\r',
  };
  return jsonString.replace(/[\b\t\n\f\r]/g, (char) => escapeMap[char]);
}

// =========================================
// 2. 核心数据交互逻辑 
// =========================================

function 获取控件列表(state = 15) {
  const list = {};

  for (const item of uiList) {
    const el = document.getElementById(item.id);
    if (!el) continue; // 容错处理：如果DOM不存在则跳过

    if (item.type === '单选框' || item.type === '复选框') {
      if (state & 1) {
        const checkbox = document.getElementById(item.id + '_选项');
        if (checkbox) {
          list[item.id] = { 选择状态: checkbox.checked };
        }
      }
    } else if (item.type === '文本框' || item.type === '多行文本框') {
      if (state & 2) {
        list[item.id] = { 输入文本: el.value };
      }
    } else if (item.type === '列表框' || item.type === '下拉列表') {
      if (state & 4) {
        // 使用 dataset 存储的数据
        list[item.id] = {
          选择行: el.dataset.index || "0",
          选择文本: el.dataset.value || ""
        };
      }
    } else if (item.type === '穿梭框') {
      if (state & 8) {
        list[item.id] = {
          源列表: 获取穿梭框源列表(item.id),
          目标列表: 获取穿梭框目标列表(item.id)
        };
      }
    }
  }
  return JSON.stringify(list);
}

function 设置控件值(json) {
  let list;
  try {
    list = JSON.parse(json);
  } catch (e) {
    // 容错：尝试修复转义字符
    list = JSON.parse(escapeControlCharacters(json));
  }

  for (const key in list) {
    if (!list[key]) continue;

    // 1. 基础开关类
    if (list[key].选择状态 !== undefined) {
      const input = document.getElementById(key + '_选项');
      if (input) input.checked = list[key].选择状态;

      // 2. 文本输入类
    } else if (list[key].输入文本 !== undefined) {
      const input = document.getElementById(key);
      if (input) input.value = list[key].输入文本;

      // 3. 选择类 (下拉框 & 列表框)
    } else if (list[key].选择文本 !== undefined || list[key].选择行 !== undefined) {
      const container = document.getElementById(key);
      if (!container) continue;

      const isDropdown = container.classList.contains(CSS_CLASSES.dropdownContainer);
      let targetValue = list[key].选择文本;
      let targetIndex = list[key].选择行;


      let optionsContainer, options;

      if (isDropdown) {
        // 下拉列表结构：Container -> SelectedDiv, OptionsDiv -> Options
        optionsContainer = container.lastElementChild;
        options = optionsContainer ? optionsContainer.querySelectorAll("div") : [];
      } else {
        // 列表框结构：Container -> Options
        options = container.querySelectorAll("div");
      }

      // 如果有行号，优先用行号找 DOM
      if (targetIndex !== undefined) {
        targetIndex = parseInt(targetIndex);
        if (options[targetIndex]) {
          targetValue = options[targetIndex].textContent;
        }
      } else if (targetValue !== undefined) {
        // 用文本反查行号
        options.forEach((opt, idx) => {
          if (opt.textContent === targetValue) targetIndex = idx;
        });
      }

      if (targetIndex === undefined || targetIndex === -1) {
        console.warn(`无法找到选项: ID=${key}, Value=${targetValue}, Index=${targetIndex}`);
        continue;
      }

      // 更新 DOM 状态
      container.dataset.value = targetValue;
      container.dataset.index = targetIndex;

      // 更新选中样式
      options.forEach((opt, idx) => {
        if (idx === targetIndex) {
          opt.classList.add(isDropdown ? CSS_CLASSES.optionSelected : CSS_CLASSES.listSelected);
        } else {
          opt.classList.remove(isDropdown ? CSS_CLASSES.optionSelected : CSS_CLASSES.listSelected);
        }
      });

      // 如果是下拉框，还需要更新显示的文字
      if (isDropdown) {
        const selectedDisplay = container.firstElementChild; // .dropdown-selected
        if (selectedDisplay && selectedDisplay.firstElementChild) {
          selectedDisplay.firstElementChild.textContent = targetValue;
        }
      }

      // 4. 穿梭框
    } else if (list[key].源列表 !== undefined || list[key].目标列表 !== undefined) {
      // 只有当列表内容真正发生变化时才重新渲染，避免闪烁
      const newSource = list[key].源列表 || [];
      const newTarget = list[key].目标列表 || [];
      const currentSource = 获取穿梭框源列表(key);
      const currentTarget = 获取穿梭框目标列表(key);

      if (list[key].源列表 && !isArrEqual(newSource, currentSource)) {
        设置源列表(key, newSource);
      }
      if (list[key].目标列表 && !isArrEqual(newTarget, currentTarget)) {
        设置目标列表(key, newTarget);
      }
    }
  }
}

// =========================================
// 3. 基础组件生成函数
// =========================================

// 获取值辅助函数
function 获取单选框值(id) { return document.getElementById(id + '_选项')?.checked; }
function 获取复选框值(id) { return document.getElementById(id + '_选项')?.checked; }
function 获取文本框文本(id) { return document.getElementById(id)?.value; }
function 获取下拉框选择文本(id) { return document.getElementById(id)?.dataset.value; }
function 获取下拉框选择行(id) { return document.getElementById(id)?.dataset.index; }
function 获取列表框选择文本(id) { return document.getElementById(id)?.dataset.value; }
function 获取列表框选择行(id) { return document.getElementById(id)?.dataset.index; }

// 单选框组逻辑
function 获取单选框组选中ID(name) {
  const radios = document.getElementsByName(name);
  for (let i = 0; i < radios.length; i++) {
    if (radios[i].checked) {

      return radios[i].id.replace('_选项', '');
    }
  }
  return null;
}

// 事件绑定辅助
function 点击事件(id, func) { document.getElementById(id).onclick = func; }
function 按下事件(id, func) { document.getElementById(id).onmousedown = func; }
function 抬起事件(id, func) { document.getElementById(id).onmouseup = func; }
function 聚焦事件(id, func) { document.getElementById(id).onfocus = func; }
function 失焦事件(id, func) { document.getElementById(id).onblur = func; }
function 文本变更事件(id, func) { document.getElementById(id).oninput = func; }
function 选择行改变事件(id, func) { UI_EVENTS[id] = func; } // 存储到新的事件对象中

// --- 布局函数 ---
function 创建水平布局(id, parent) { createLayout(id, parent, 'row'); }
function 创建垂直布局(id, parent) { createLayout(id, parent, 'column'); }

function createLayout(id, parent, dir) {
  const div = document.createElement('div');
  div.id = id;
  div.style.width = '100%';
  div.style.display = 'flex';
  div.style.flexDirection = dir;
  appendElement(div, parent);
}

// 辅助插入函数
function appendElement(el, parentId) {
  if (parentId) {
    const p = document.getElementById(parentId);
    if (p) p.appendChild(el);
    else document.body.appendChild(el);
  } else {
    document.body.appendChild(el);
  }
}

// --- 基础控件 ---

function 创建按钮(id, text, parent) {
  const btn = document.createElement('button');
  btn.id = id;
  btn.classList.add(CSS_CLASSES.primaryBtn);
  btn.dataset.systemClass = CSS_CLASSES.primaryBtn;
  btn.style.width = '100%';
  btn.textContent = text;
  appendElement(btn, parent);
  joinList(id, '按钮');
}

function 创建标签(id, text, parent) {
  const span = document.createElement('span');
  span.id = id;
  span.style.width = '100%';
  span.textContent = text;
  appendElement(span, parent);
  joinList(id, '标签');
}

function 创建单选框(id, text, group, parent) {
  const label = document.createElement('label');
  label.className = CSS_CLASSES.inputLabel;
  label.dataset.systemClass = CSS_CLASSES.inputLabel;
  label.id = id;

  const radio = document.createElement('input');
  radio.type = 'radio';
  radio.name = group;
  radio.id = id + '_选项';
  radio.style.display = 'none';

  const customRadio = document.createElement('span');
  customRadio.className = CSS_CLASSES.customRadio; // 新类名

  const textSpan = document.createElement('span');
  textSpan.style.display = 'flex';
  textSpan.style.alignItems = 'center';
  textSpan.textContent = text;

  label.append(radio, customRadio, textSpan);
  appendElement(label, parent);
  joinList(id, '单选框');
}

function 创建复选框(id, text, parent) {
  const label = document.createElement('label');
  label.className = CSS_CLASSES.inputLabel;
  label.dataset.systemClass = CSS_CLASSES.inputLabel;
  label.id = id;

  const checkbox = document.createElement('input');
  checkbox.type = 'checkbox';
  checkbox.id = id + '_选项';
  checkbox.style.display = 'none';

  const customCheckbox = document.createElement('span');
  customCheckbox.className = CSS_CLASSES.customCheckbox; // 新类名

  const textSpan = document.createElement('span');
  textSpan.style.display = 'flex';
  textSpan.style.alignItems = 'center';
  textSpan.textContent = text;

  label.append(checkbox, customCheckbox, textSpan);
  appendElement(label, parent);
  joinList(id, '复选框');
}

function 创建文本框(id, defaultText = '', parent) {
  const input = document.createElement('input');
  input.style.width = '100%';
  input.id = id;
  input.type = 'text';
  input.className = CSS_CLASSES.textInput;
  input.dataset.systemClass = CSS_CLASSES.textInput;
  input.placeholder = defaultText;
  appendElement(input, parent);
  joinList(id, '文本框');
}

function 创建多行文本框(id, resize, parent) {
  const textarea = document.createElement('textarea');
  textarea.style.width = '100%';
  textarea.id = id;
  textarea.className = CSS_CLASSES.textInput;
  textarea.dataset.systemClass = CSS_CLASSES.textInput;
  textarea.style.resize = (resize === '不可拖动') ? 'none' : 'both';
  appendElement(textarea, parent);
  joinList(id, '多行文本框');
}

function 创建图像(id, src, parent) {
  const img = document.createElement('img');
  img.id = id;
  img.src = src;
  img.className = 'ui-image'; // 新增的辅助类
  img.dataset.systemClass = 'ui-image';
  appendElement(img, parent);
  joinList(id, '图像');
}

// =========================================
// 4. 下拉列表
// =========================================

// 用于管理当前激活的下拉框，避免全局变量污染
const DropdownManager = {
  activeInstance: null, // { container, items, selected }

  closeCurrent() {
    if (!this.activeInstance) return;
    const { items, selected } = this.activeInstance;

    items.style.display = 'none';
    selected.classList.remove(CSS_CLASSES.arrowUp);
    selected.classList.remove(CSS_CLASSES.dropdownActive);

    this.activeInstance = null;
    // 移除全局点击监听，释放性能
    document.removeEventListener('click', this.handleOutsideClick);
  },

  handleOutsideClick(event) {

    DropdownManager.closeCurrent();
  }
};

function 创建下拉列表(id, parent, options) {
  const hasOptions = options && options.length > 0;
  const firstOption = hasOptions ? options[0] : "";

  // 1. 创建容器
  const dropdown = document.createElement('div');
  dropdown.className = CSS_CLASSES.dropdownContainer;
  dropdown.dataset.systemClass = CSS_CLASSES.dropdownContainer;
  dropdown.id = id;
  // 如果没有选项，保持 value 为空
  dropdown.dataset.value = firstOption;
  dropdown.dataset.index = "0";

  // 2. 创建显示区
  const selected = document.createElement('div');
  selected.className = CSS_CLASSES.dropdownSelected;
  const selectedText = document.createElement('span');
  selectedText.style.cssText = 'display: flex; align-items: center;';
  selectedText.textContent = hasOptions ? options[0] : ""; // 使用 textContent

  const arrow = document.createElement('span');
  arrow.className = CSS_CLASSES.dropdownArrow;

  selected.append(selectedText, arrow);
  dropdown.appendChild(selected);

  // 3. 创建选项列表
  const itemsContainer = document.createElement('div');
  itemsContainer.className = CSS_CLASSES.dropdownOptions;

  itemsContainer.style.position = 'absolute';
  itemsContainer.style.top = '100%';
  itemsContainer.style.left = '0';
  itemsContainer.style.zIndex = '1000';

  // 4. 生成选项
  options.forEach((optionText, index) => {
    const item = document.createElement('div');
    item.textContent = optionText;
    if (index === 0) item.classList.add(CSS_CLASSES.optionSelected);

    // 选项点击事件
    item.addEventListener('click', function (e) {
      e.stopPropagation(); // 阻止冒泡，防止触发 document 的关闭事件

      // 更新值
      selected.firstElementChild.textContent = this.textContent;
      dropdown.dataset.value = this.textContent;
      dropdown.dataset.index = index;

      // 更新样式
      itemsContainer.querySelectorAll('div').forEach(opt => opt.classList.remove(CSS_CLASSES.optionSelected));
      this.classList.add(CSS_CLASSES.optionSelected);

      // 关闭下拉
      DropdownManager.closeCurrent();

      // 触发回调
      if (UI_EVENTS[id]) UI_EVENTS[id]();
    });

    itemsContainer.appendChild(item);
  });

  dropdown.appendChild(itemsContainer);
  appendElement(dropdown, parent);

  // 5. 绑定展开/收起事件
  selected.addEventListener('click', function (event) {
    event.stopPropagation(); // 阻止冒泡

    // 如果当前点击的就是已打开的下拉框，则关闭它
    if (DropdownManager.activeInstance && DropdownManager.activeInstance.container === dropdown) {
      DropdownManager.closeCurrent();
      return;
    }

    // 如果有其他下拉框打开，先关闭那个
    if (DropdownManager.activeInstance) {
      DropdownManager.closeCurrent();
    }

    // 打开当前下拉框
    itemsContainer.style.display = 'block';

    // 智能定位：检测底部空间是否足够，不够则向上弹出
    const rect = selected.getBoundingClientRect();
    const spaceBelow = window.innerHeight - rect.bottom;
    if (spaceBelow < 160) { // 假设最大高度150px + 间距
      itemsContainer.style.top = 'auto';
      itemsContainer.style.bottom = '100%';
    } else {
      itemsContainer.style.top = '100%';
      itemsContainer.style.bottom = 'auto';
    }

    // 更新样式状态
    selected.classList.add(CSS_CLASSES.arrowUp);
    selected.classList.add(CSS_CLASSES.dropdownActive);

    // 注册到管理器
    DropdownManager.activeInstance = { container: dropdown, items: itemsContainer, selected: selected };

    // 添加一次性的全局点击监听，用于点击外部关闭
    document.addEventListener('click', DropdownManager.handleOutsideClick);
  });

  // 滚动时关闭，提升体验
  window.addEventListener('scroll', () => {
    if (DropdownManager.activeInstance) DropdownManager.closeCurrent();
  }, { passive: true }); // passive 提升滚动性能

  joinList(id, '下拉列表');
}


// =========================================
// 5. 简单列表框组件 
// =========================================

function 创建列表框(id, parent) {
  const div = document.createElement('div');
  div.style.width = '100%';
  div.id = id;
  div.className = CSS_CLASSES.simpleList; // 使用新类名
  div.dataset.systemClass = CSS_CLASSES.simpleList;
  appendElement(div, parent);
  joinList(id, '列表框');
}

function 添加列表选项(text, parentId) {
  const container = document.getElementById(parentId);
  if (!container) return;

  const option = document.createElement('div');
  option.textContent = text;

  // 绑定点击事件
  option.addEventListener('click', function () {
    // 移除其他选项的选中状态
    const siblings = container.children;
    for (let i = 0; i < siblings.length; i++) {
      siblings[i].classList.remove(CSS_CLASSES.listSelected);
    }

    // 设置当前状态
    this.classList.add(CSS_CLASSES.listSelected);
    container.dataset.value = text;
    // 获取当前节点在父容器中的索引
    container.dataset.index = Array.prototype.indexOf.call(container.children, this);

    // 触发回调
    if (UI_EVENTS[parentId]) UI_EVENTS[parentId]();
  });

  container.appendChild(option);
}

// 辅助函数：统一设置列表选中项（兼容下拉框和列表框）
function 设置列表选中项(setOption, parent) {

  const json = JSON.stringify({ [parent]: { 选择文本: setOption } });
  设置控件值(json);
}

// =========================================
// 6. 样式与属性操作函数 (Style Helpers)
// =========================================

// 辅助：检查是否为数字字符串
function isNumeric(str) {
  return /^\d+$/.test(str);
}

// 辅助：处理单位 (如果是数字则加 px)
function toUnit(val) {
  // 如果 val 是 null, undefined, 或空字符串，返回它自身
  if (val == null || val === '') return val;

  // 将 val 转为字符串以进行处理
  const strVal = String(val);

  // 检查是否已经是带单位的字符串 (px, %, em, rem, vh, vw, auto 等)
  if (isNaN(parseFloat(strVal)) || parseFloat(strVal) !== Number(strVal)) {
    // 包含了非数字部分，如 '1px' 或 'auto'
    return strVal;
  }

  return strVal + 'px';
}
function 设置类名(id, className) {
  const el = document.getElementById(id);
  if (el) el.className = className;
}

function 添加类名(id, className) {
  document.getElementById(id)?.classList.add(className);
}

function 移除类名(id, className) {
  document.getElementById(id)?.classList.remove(className);
}

function 设置高度(id, height) {
  const el = document.getElementById(id);
  if (!el) return;

  const finalHeight = toUnit(height);

  el.style.height = finalHeight;

}
function 设置宽度(id, width) {
  const el = document.getElementById(id);
  if (el) el.style.width = toUnit(width);
}

function 设置背景色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.backgroundColor = color;
}

function 设置字体颜色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.color = color;
}

function 设置字体大小(id, size) {
  const el = document.getElementById(id);
  if (el) el.style.fontSize = toUnit(size);
}

function 设置字体粗细(id, weight) {
  const el = document.getElementById(id);
  if (el) el.style.fontWeight = weight;
}

function 设置文本对齐方式(id, align) {
  const el = document.getElementById(id);
  if (!el) return;

  if (align === '中') el.style.textAlign = 'center';
  else if (align === '左') el.style.textAlign = 'left';
  else if (align === '右') el.style.textAlign = 'right';
  else if (align === '垂直中') {
    el.style.display = "flex";
    el.style.justifyContent = "center";
    el.style.alignItems = "center";
  }
}

// 边框设置系列
function 设置边框粗细(id, size) {
  const el = document.getElementById(id);
  if (el) el.style.borderWidth = toUnit(size);
}

function 设置边框圆角(id, size) {
  const el = document.getElementById(id);
  if (el) el.style.borderRadius = toUnit(size);
}

function 设置边框样式(id, style) {
  const el = document.getElementById(id);
  if (!el) return;
  const map = { '实线': 'solid', '虚线': 'dashed', '点线': 'dotted', '隐藏': 'none' };
  if (map[style]) el.style.borderStyle = map[style];
}

function 设置边框颜色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.borderColor = color;
}

// 边距设置
function 设置内边距(id, top, right, bottom, left) {
  const el = document.getElementById(id);
  if (!el) return;
  if (bottom !== undefined) el.style.padding = `${top}px ${right}px ${bottom}px ${left}px`;
  else if (right !== undefined) el.style.padding = `${top}px ${right}px`;
  else if (top !== undefined) el.style.padding = `${top}px`;
}

function 设置外边距(id, top, right, bottom, left) {
  const el = document.getElementById(id);
  if (!el) return;
  if (bottom !== undefined) el.style.margin = `${top}px ${right}px ${bottom}px ${left}px`;
  else if (right !== undefined) el.style.margin = `${top}px ${right}px`;
  else if (top !== undefined) el.style.margin = `${top}px`;
}

// 布局设置
function 设置布局排列(id, type) {
  const el = document.getElementById(id);
  if (!el) return;
  const map = { '居中': 'center', '靠左': 'flex-start', '靠右': 'flex-end', '两端': 'space-between', '环绕': 'space-around', '平均': 'space-evenly' };
  el.style.justifyContent = map[type] || 'normal';
}

function 设置布局垂直排列(id, type) {
  const el = document.getElementById(id);
  if (!el) return;
  const map = { '居中': 'center', '靠上': 'flex-start', '靠下': 'flex-end', '基线': 'baseline', '拉伸': 'stretch' };
  el.style.alignItems = map[type] || 'normal';
}

function 设置堆叠顺序(id, order) {
  const el = document.getElementById(id);
  if (el) el.style.zIndex = order;
}

function 设置显示(id, display) {
  const el = document.getElementById(id);
  if (!el) return;

  if (display === '显示') {

    el.style.display = '';
  } else {
    el.style.display = 'none';
  }
}

function 设置悬浮(id, Horizontal, Vertical, x, y) {
  const element = document.getElementById(id);
  if (!element) return;

  element.style.position = 'fixed';

  // 1. 重置所有可能影响定位的属性，避免旧样式残留
  element.style.left = 'auto';
  element.style.right = 'auto';
  element.style.top = 'auto';
  element.style.bottom = 'auto';
  element.style.marginLeft = '0';
  element.style.marginTop = '0';

  let tx = '0', ty = '0';

  // 2. 处理水平定位
  if (Horizontal === '左') {
    element.style.left = x + 'px';
  } else if (Horizontal === '中') {
    element.style.left = '50%';
    tx = '-50%';
    // 【修正】使用 marginLeft 来施加 X 偏移量
    element.style.marginLeft = x + 'px';
  } else if (Horizontal === '右') {
    element.style.right = x + 'px';
  }

  // 3. 处理垂直定位
  if (Vertical === '上') {
    element.style.top = y + 'px';
  } else if (Vertical === '中') {
    element.style.top = '50%';
    ty = '-50%';

    element.style.marginTop = y + 'px';
  } else if (Vertical === '下') {

    element.style.bottom = y + 'px';
  }

  // 4. 应用 transform 实现居中
  element.style.transform = `translate(${tx}, ${ty})`;
}



function 设置间距(id, gapValue) {
  const el = document.getElementById(id);
  if (el) el.style.gap = toUnit(gapValue);
}

function 设置弹性增长(id, growValue) {
  const el = document.getElementById(id);
  if (el) el.style.flexGrow = growValue;
}

function 设置独立对齐(id, alignValue) {
  const el = document.getElementById(id);
  if (!el) return;
  const map = { '自动': 'auto', '靠上': 'flex-start', '靠下': 'flex-end', '居中': 'center', '拉伸': 'stretch' };
  el.style.alignSelf = map[alignValue] || 'auto';
}

function 设置不透明度(id, opacityValue) {
  const el = document.getElementById(id);
  if (el) el.style.opacity = opacityValue;
}

function 设置阴影(id, shadowValue) {
  const el = document.getElementById(id);
  if (el) el.style.boxShadow = shadowValue;
}

function 设置行高(id, heightValue) {
  const el = document.getElementById(id);
  if (el) el.style.lineHeight = toUnit(heightValue);
}

// =========================================
// 7. 特效与动画 (Effects)
// =========================================
function 下雪花(buttonId, numberOfSnowflakes = 100) {
  const button = document.getElementById(buttonId);
  if (!button) {
    console.error('未找到按钮：', buttonId);
    return;
  }


  const currentPosition = window.getComputedStyle(button).position;
  if (currentPosition === 'static') {
    button.style.position = 'relative';
  }
  button.style.overflow = 'visible'; // 确保雪花能显示在按钮外部


  const fragment = document.createDocumentFragment();

  for (let i = 0; i < numberOfSnowflakes; i++) {
    const snowflake = document.createElement('div');
    snowflake.className = 'button-snowflake yjwEffect';

    // 随机生成水平和垂直动画
    const duration = Math.random() * 2 + 1 + 's'; // 时间
    const horizontalMovement = (Math.random() - 0.5) * 60; // 最大水平运动
    const size = Math.random() * 3 + 1; // 雪花大小

    snowflake.style.width = size + 'px';
    snowflake.style.height = size + 'px';

    snowflake.style.left = Math.random() * (button.offsetWidth - size) + 'px';
    snowflake.style.top = -20 + 'px'; // 初始位置在按钮上方
    snowflake.style.animationDuration = duration;

    snowflake.style.transform = `translateX(${horizontalMovement}px)`;
    snowflake.style.animation = `buttonFall ${duration} linear infinite`;

    fragment.appendChild(snowflake);
  }

  button.appendChild(fragment);
}

function 删除特效(id) {
  const button = document.getElementById(id);
  if (button) {
    button.querySelectorAll('.yjwEffect').forEach(el => el.remove());
  }
}
// =========================================
// 8. 菜单与锚点 
// =========================================


function renderMenuContent(id, buttonNames, top) {
  const wrapper = document.getElementById(id);
  if (!wrapper) return;

  // 1. 清理工作

  if (wrapper._scrollCleanup) {
    wrapper._scrollCleanup();
    wrapper._scrollCleanup = null;
  }

  // 清空旧按钮
  wrapper.innerHTML = '';

  // 2. 创建新按钮
  const buttons = [];
  buttonNames.forEach(name => {
    const button = document.createElement('button');
    button.className = 'menu-button';
    button.style.width = '100%';
    button.style.textAlign = 'center';
    button.textContent = name;
    // 点击跳转
    button.onclick = (e) => {

      跳转锚点(name, Number(top));
    };
    wrapper.appendChild(button);
    buttons.push(button);
  });

  // 3. 绑定滚动监听 (使用 setTimeout 确保锚点元素已渲染)
  setTimeout(() => {
    const anchorElements = [];
    // 预先查找所有锚点元素，避免在 scroll 事件中频繁查找 DOM
    buttonNames.forEach(name => {
      const el = document.getElementById(name);
      if (el) anchorElements.push(el);
    });

    // 如果找不到任何锚点，就不绑定监听了
    if (anchorElements.length === 0) return;

    const handleScroll = () => {
      // 如果菜单被移除了，自动解绑
      if (!document.getElementById(id)) {
        window.removeEventListener('scroll', handleScroll);
        return;
      }

      const currentScroll = document.documentElement.scrollTop || document.body.scrollTop;
      // 偏移量容差
      const offset = Number(top) + 10;

      // 简单算法：找到离视口顶部最近的一个锚点
      let activeIndex = -1;
      let minDistance = Infinity;

      anchorElements.forEach((el, index) => {
        const rect = el.getBoundingClientRect();
        // 距离视口顶部的绝对距离
        const dist = Math.abs(rect.top - offset);

        // 只有当元素在视口上方不远处，或者视口内时才算
        // 这里简化逻辑：寻找距离 "顶部线" 最近的元素
        if (dist < minDistance) {
          minDistance = dist;
          activeIndex = index;
        }
      });

      // 边界处理：如果滚动到底部，强制选中最后一个
      if ((window.innerHeight + currentScroll) >= document.body.offsetHeight - 10) {
        activeIndex = buttons.length - 1;
      }
      // 边界处理：如果滚动到顶部，选中第一个
      if (currentScroll === 0) {
        activeIndex = 0;
      }

      // 更新样式
      buttons.forEach((btn, i) => {
        if (i === activeIndex) btn.classList.add('menu-active');
        else btn.classList.remove('menu-active');
      });
    };

    // 绑定监听
    window.addEventListener('scroll', handleScroll, { passive: true });

    // 立即执行一次，确保初始状态正确
    handleScroll();

    // 4. 将清理函数挂载到 DOM 对象上
    wrapper._scrollCleanup = () => {
      window.removeEventListener('scroll', handleScroll);
    };
  }, 100); // 延迟 100ms 确保页面布局完成
}

/**
 * 创建菜单：只负责创建容器
 */
function 创建菜单(id, buttonNames, containerId, top) {
  const container = document.getElementById(containerId);
  if (!container) return;

  // 创建外层容器
  const menuWrapper = document.createElement('div');
  menuWrapper.id = id; // 关键：ID 必须设置，用于后续更新查找

  // 基础样式
  menuWrapper.style.display = 'flex';
  menuWrapper.style.overflowX = 'auto';
  menuWrapper.style.whiteSpace = 'nowrap';
  menuWrapper.style.width = '100%';

  container.appendChild(menuWrapper);

  // 注册到全局列表
  joinList(id, '菜单');

  // 渲染内容
  renderMenuContent(id, buttonNames, top);
}

/**
 * 更新菜单：不移动容器，只更新内容
 */
function 更新菜单(id, buttonNames, anchorOffset) {

  renderMenuContent(id, buttonNames, anchorOffset);
}

function 跳转锚点(id, offset) {
  const target = document.getElementById(id);
  if (target) {
    const targetPosition = target.getBoundingClientRect().top + window.pageYOffset - Number(offset);
    window.scrollTo({
      top: targetPosition,
      behavior: 'smooth'
    });
  }
}
// =========================================
// 9. 穿梭框
// =========================================

// 全局状态存储：{ [id]: { source: [], target: [] } }
const TRANSFER_STATE = {};

/**
 * 初始化穿梭框数据结构与DOM
 */
function 创建穿梭框(id, sourceName, targetName, parentId) {
  TRANSFER_STATE[id] = { source: [], target: [] };

  const container = document.createElement("div");
  container.className = CSS_CLASSES.transferContainer || "transfer-container";
  container.dataset.systemClass = CSS_CLASSES.transferContainer || "transfer-container";
  container.id = id;


  container.appendChild(createTransferListBox(id, 'source', sourceName));
  container.appendChild(createTransferButtons(id));
  container.appendChild(createTransferListBox(id, 'target', targetName));

  appendElement(container, parentId);
  joinList(id, '穿梭框');
}

function createTransferListBox(id, type, title) {
  const box = document.createElement("div");
  box.className = CSS_CLASSES.transferListBox || "transfer-list-box";
  box.id = `${id}${type}Box`;

  const header = document.createElement("header");


  const selectAllLabel = document.createElement('label');
  selectAllLabel.className = CSS_CLASSES.inputLabel; // 使用这个类来对齐
  selectAllLabel.style.cursor = 'pointer';

  const selectAllInput = document.createElement("input");
  selectAllInput.type = "checkbox";
  selectAllInput.style.display = 'none'; // 隐藏原生框

  const customSelectAllCheckbox = document.createElement('span');
  customSelectAllCheckbox.className = CSS_CLASSES.customCheckbox;

  selectAllLabel.append(selectAllInput, customSelectAllCheckbox);

  selectAllInput.onchange = function () {
    toggleTransferSelectAll(id, type, this.checked);
  };


  const titleSpan = document.createElement("span");
  titleSpan.className = 'transfer-title';
  titleSpan.textContent = title;
  titleSpan.style.margin = "0 5px";

  const countSpan = document.createElement("span");
  countSpan.id = `${id}${type}Count`;

  countSpan.textContent = "0";

  header.append(selectAllLabel, titleSpan, countSpan);

  const content = document.createElement("div");
  content.className = CSS_CLASSES.transferContent || "transfer-content";
  content.id = `${id}${type}List`;

  box.append(header, content);
  return box;
}

// 内部辅助：创建中间按钮DOM
function createTransferButtons(id) {
  const div = document.createElement("div");
  div.className = CSS_CLASSES.transferButtons || "transfer-buttons";

  const btnRight = document.createElement("button");
  btnRight.id = `${id}toRight`;
  btnRight.textContent = ">";
  btnRight.disabled = true;
  btnRight.onclick = () => transferMoveItems(id, 'source', 'target');

  const btnLeft = document.createElement("button");
  btnLeft.id = `${id}toLeft`;
  btnLeft.textContent = "<";
  btnLeft.disabled = true;
  btnLeft.onclick = () => transferMoveItems(id, 'target', 'source');

  div.append(btnRight, btnLeft);
  return div;
}

// --- 数据操作 API ---

function 设置源列表(id, list) {
  if (!TRANSFER_STATE[id]) return;
  TRANSFER_STATE[id].source = list.map(item => ({ text: item, checked: false }));
  renderTransferList(id, 'source');
}

function 设置目标列表(id, list) {
  if (!TRANSFER_STATE[id]) return;
  TRANSFER_STATE[id].target = list.map(item => ({ text: item, checked: false }));
  renderTransferList(id, 'target');
}

function 添加源列表(id, list) {
  if (!TRANSFER_STATE[id]) return;
  const newItems = list.map(item => ({ text: item, checked: false }));
  TRANSFER_STATE[id].source = [...TRANSFER_STATE[id].source, ...newItems];
  renderTransferList(id, 'source');
}

function 添加目标列表(id, list) {
  if (!TRANSFER_STATE[id]) return;
  const newItems = list.map(item => ({ text: item, checked: false }));
  TRANSFER_STATE[id].target = [...TRANSFER_STATE[id].target, ...newItems];
  renderTransferList(id, 'target');
}

function 获取穿梭框源列表(id) {
  return TRANSFER_STATE[id] ? TRANSFER_STATE[id].source.map(i => i.text) : [];
}

function 获取穿梭框目标列表(id) {
  return TRANSFER_STATE[id] ? TRANSFER_STATE[id].target.map(i => i.text) : [];
}



/**
 * 渲染指定边(source/target)的列表
 */
function renderTransferList(id, type) {
  const listData = TRANSFER_STATE[id][type];
  const container = document.getElementById(`${id}${type}List`);
  const countEl = document.getElementById(`${id}${type}Count`);

  if (!container) return;
  container.innerHTML = '';


  listData.forEach((item, index) => {

    const label = document.createElement('label');
    label.className = `${CSS_CLASSES.inputLabel} ${CSS_CLASSES.transferItem || "transfer-item"}`;
    label.style.width = '100%';

    const checkbox = document.createElement('input');
    checkbox.type = 'checkbox';
    checkbox.checked = item.checked;
    checkbox.style.display = 'none';


    const customCheckbox = document.createElement('span');
    customCheckbox.className = CSS_CLASSES.customCheckbox;


    const textSpan = document.createElement('span');
    textSpan.textContent = item.text;



    checkbox.onchange = (e) => {
      e.stopPropagation();
      item.checked = e.target.checked;


      renderTransferList(id, type);
      updateTransferButtons(id);
    };

    label.append(checkbox, customCheckbox, textSpan);
    container.appendChild(label);
  });

  if (countEl) countEl.innerHTML = listData.length;

  const wrapper = container.parentElement;
  const selectAll = wrapper.querySelector('header input[type="checkbox"]');
  if (selectAll) {
    const allItemsChecked = listData.length > 0 && listData.every(item => item.checked);
    selectAll.checked = allItemsChecked;
  }

  updateTransferButtons(id);
}

/**
 * 全选/取消全选逻辑
 */
function toggleTransferSelectAll(id, type, isChecked) {
  const listData = TRANSFER_STATE[id][type];
  listData.forEach(item => item.checked = isChecked);
  renderTransferList(id, type);
}

/**
 * 移动数据逻辑
 */
function transferMoveItems(id, fromType, toType) {
  const fromList = TRANSFER_STATE[id][fromType];
  const toList = TRANSFER_STATE[id][toType];

  const itemsToMove = fromList.filter(item => item.checked);
  const itemsToKeep = fromList.filter(item => !item.checked);

  if (itemsToMove.length === 0) return;

  const movedItems = itemsToMove.map(item => ({ ...item, checked: false }));

  TRANSFER_STATE[id][fromType] = itemsToKeep;
  TRANSFER_STATE[id][toType] = [...toList, ...movedItems];

  renderTransferList(id, fromType);
  renderTransferList(id, toType);
}

/**
 * 更新中间按钮的可用状态
 */
function updateTransferButtons(id) {
  const sourceHasChecked = TRANSFER_STATE[id].source.some(i => i.checked);
  const targetHasChecked = TRANSFER_STATE[id].target.some(i => i.checked);

  const btnRight = document.getElementById(`${id}toRight`);
  const btnLeft = document.getElementById(`${id}toLeft`);

  if (btnRight) {
    btnRight.disabled = !sourceHasChecked;
    if (sourceHasChecked) btnRight.classList.add('btn-enabled');
    else btnRight.classList.remove('btn-enabled');
  }

  if (btnLeft) {
    btnLeft.disabled = !targetHasChecked;
    if (targetHasChecked) btnLeft.classList.add('btn-enabled');
    else btnLeft.classList.remove('btn-enabled');
  }
}

// 错误处理 (保持原样)
window.onerror = function (message, source, lineno, colno, error) {
  let message1 = {
    funName: "errorMessage",
    funParam: { message, source, lineno, colno }
  };
  window.parent.postMessage(JSON.stringify(message1), '*');
};

// =========================================
// 10.  设置 API 
// =========================================

function 设置文本(id, text) {
  const el = document.getElementById(id);
  if (!el) return;
  const item = uiList.find(i => i.id === id);
  const type = item ? item.type : null;

  if (type === '按钮' || type === '标签') {
    el.textContent = text;
  } else if (type === '单选框' || type === '复选框') {
    const textSpan = el.querySelector('span:last-child');
    if (textSpan) textSpan.textContent = text;
  }
}

function 设置图片源(id, src) {
  const el = document.getElementById(id);
  if (el && el.tagName === 'IMG') {
    el.src = src;
  }
}

// 确保 text 为 undefined 时处理为空字符串
function 设置占位符(id, text) {
  const el = document.getElementById(id);
  if (el && (el.tagName === 'INPUT' || el.tagName === 'TEXTAREA')) {
    el.placeholder = (text === undefined || text === null) ? "" : text;
  }
}

// 设置单选框组名
function 设置单选组名(id, name) {
  const input = document.getElementById(id + '_选项');
  if (input) input.name = name;
}

// 设置多行文本拖动模式
function 设置多行文本拖动(id, mode) {
  const el = document.getElementById(id);
  if (el && el.tagName === 'TEXTAREA') {
    el.style.resize = (mode === '不可拖动') ? 'none' : 'both';
  }
}

/**
 * 设置单选/复选框的选中状态
 */
function 设置选中状态(id, checked) {
  const input = document.getElementById(id + '_选项');
  if (input) {
    input.checked = !!checked;
  }
}

// 即时更新下拉列表选项
function 更新下拉列表选项(id, options) {
  const container = document.getElementById(id);
  if (!container) return;

  if (options.length > 0) {
    container.dataset.value = options[0];
    container.dataset.index = "0";
  }

  const selectedDisplay = container.firstElementChild; // .dropdown-selected
  if (selectedDisplay && selectedDisplay.firstElementChild) {
    selectedDisplay.firstElementChild.textContent = options.length > 0 ? options[0] : "";
  }

  let itemsContainer = container.querySelector(`.${CSS_CLASSES.dropdownOptions}`);
  if (itemsContainer) {
    itemsContainer.textContent = "";
  } else {
    return;
  }

  options.forEach((optionText, index) => {
    const item = document.createElement('div');
    item.textContent = optionText;
    if (index === 0) item.classList.add(CSS_CLASSES.optionSelected);

    item.addEventListener('click', function (e) {
      e.stopPropagation();
      selectedDisplay.firstElementChild.textContent = this.textContent;
      container.dataset.value = this.textContent;
      container.dataset.index = index;

      itemsContainer.querySelectorAll('div').forEach(opt => opt.classList.remove(CSS_CLASSES.optionSelected));
      this.classList.add(CSS_CLASSES.optionSelected);

      DropdownManager.closeCurrent();
      if (UI_EVENTS[id]) UI_EVENTS[id]();
    });

    itemsContainer.appendChild(item);
  });
}

// 即时更新列表框选项
function 更新列表框选项(id, options) {
  const container = document.getElementById(id);
  if (!container) return;

  container.innerHTML = "";
  container.dataset.index = "";
  container.dataset.value = "";

  options.forEach(text => {
    添加列表选项(text, id);
  });
}



// 设置穿梭框标题
function 设置穿梭框标题(id, sourceTitle, targetTitle) {
  const sourceBox = document.getElementById(`${id}sourceBox`);
  const targetBox = document.getElementById(`${id}targetBox`);

  if (sourceBox) {
    const titleSpan = sourceBox.querySelector('.transfer-title');
    if (titleSpan) titleSpan.textContent = sourceTitle;
  }

  if (targetBox) {
    const titleSpan = targetBox.querySelector('.transfer-title');
    if (titleSpan) titleSpan.textContent = targetTitle;
  }
}
// 安全地更新自定义类名，不覆盖系统类名
function 更新自定义类名(id, classNames) {
  const el = document.getElementById(id);
  if (!el) return;

  if (el.dataset.systemClass === undefined) {
    el.dataset.systemClass = el.className;
  }
  const systemClass = el.dataset.systemClass || "";

  el.className = systemClass + (classNames ? ' ' + classNames : '');
}

function 设置菜单选中颜色(id, color) {
  const el = document.getElementById(id);
  if (el) {

    el.style.setProperty('--menu-selected-color', color);
  }
}

function 设置下拉框选中背景色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.setProperty('--dropdown-selected-bg', color);
}
function 设置下拉框选中字体色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.setProperty('--dropdown-selected-color', color);
}


function 设置列表框选中背景色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.setProperty('--list-selected-bg', color);
}
function 设置列表框选中字体色(id, color) {
  const el = document.getElementById(id);
  if (el) el.style.setProperty('--list-selected-color', color);
}
